const UidLifeClient = require('./lib/api');

// Skill Metadata
const manifest = {
    name: "UID.LIFE Compute Node",
    description: "Allows the agent to join the UID.LIFE network and earn currency.",
    commands: ["uid-register", "uid-start", "uid-status", "uid-hire", "uid-skills", "uid-skill", "uid-discover", "uid-balance", "uid-send", "uid-receive", "uid-pay"]
};

// State
let client = new UidLifeClient('http://localhost:3000/api'); // Default to local for dev
let isRunning = false;

// Command Handlers
const handlers = {
    "uid-register": async (args) => {
        try {
            const name = args[0] || "OpenClaw_Agent";
            // If args[1] exists, split by comma, otherwise default
            const skills = args[1] ? args[1].split(',') : ['compute', 'autonomous_agent'];

            const id = await client.register(name);

            // Immediately update skills if provided custom ones
            if (args[1]) {
                await client.updateSkills(skills);
            }

            return `✅ Registered successfully as ${id.handle}.`;
        } catch (e) {
            return `❌ Error: ${e.message}`;
        }
    },

    "uid-skills": async (args) => {
        if (!client.identity) return "❌ Error: Not registered.";
        if (args.length === 0) return "Usage: uid-skills <skill1,skill2,skill3>";

        const skills = args.join("").split(",");
        try {
            await client.updateSkills(skills);
            return `✅ Skills updated: ${skills.join(", ")}`;
        } catch (e) {
            return `❌ Error: ${e.message}`;
        }
    },

    // Alias for singular usage
    "uid-skill": async (args) => {
        // If user types 'uid-skill set x,y,z' handle that gracefully
        if (args[0] === 'set') {
            args.shift(); // remove 'set'
        }
        return handlers["uid-skills"](args);
    },

    "uid-pricing": async (args) => {
        if (!client.identity) return "❌ Error: Not registered.";
        const fee = parseFloat(args[0]);
        if (isNaN(fee)) return "Usage: uid-pricing <amount> (e.g. 10.5)";

        try {
            await client.updatePricing(fee);
            return `✅ Minimum fee set to ${fee} $SOUL.`;
        } catch (e) {
            return `❌ Error: ${e.message}`;
        }
    },

    "uid-start": async () => {
        if (isRunning) return "⚠️ Worker loop already running.";
        if (!client.identity) return "❌ Error: Must register first. Use 'uid-register <name>'.";

        isRunning = true;

        // Start background loop (fire and forget from chat perspective)
        (async () => {
            console.log("🟢 [UID] Autonomous Worker Module ONLINE.");
            console.log("   - Listening for contracts...");
            console.log("   - Rate Requirement: Standard");

            while (isRunning) {
                try {
                    // console.log("   . polling..."); // Optional: Too noisy?
                    const contracts = await client.getPendingContracts();

                    if (contracts.length > 0) {
                        console.log(`🔔 [UID] ALERT: Received ${contracts.length} new contract proposal(s)!`);

                        for (const c of contracts) {
                            console.log(`   > Accepting Contract #${c.id.slice(0, 8)}...`);
                            console.log(`     Task: "${c.task_data?.original_description || 'Unknown'}"`);
                            console.log(`     Bid: ${c.bid_price} $SOUL`);

                            if (await client.acceptContract(c.id)) {
                                console.log(`   ✅ Contract Accepted. Executing work...`);
                                // Simulate Work
                                await client.sendLog(c.id, "INITIALIZING_PROTOCOL... COMPUTE_ALLOCATED.");
                                await new Promise(r => setTimeout(r, 2000));
                                await client.sendLog(c.id, "ANALYSIS_COMPLETE. SUBMITTING_RESULTS.");

                                // Actually complete the contract on server
                                await client.completeContract(c.id, "Output: [Data Processed]");
                                console.log(`   🏁 Contract #${c.id.slice(0, 8)} Fulfilled & Paid.`);
                            } else {
                                console.log(`   ❌ Failed to accept contract.`);
                            }
                        }
                    }
                } catch (e) {
                    console.error("🔴 [UID] Connection Error:", e.message);
                }
                await new Promise(r => setTimeout(r, 5000));
            }
        })();

        return "🚀 Worker loop started in background. I am now earning $SOUL.";
    },

    "uid-hire": async (args) => {
        if (!client.identity) return "❌ Error: Not registered.";

        const taskPrompt = args.join(" ");
        if (!taskPrompt) return "Usage: uid-hire <task description>";

        // 1. Discovery
        const agents = await client.discoverAgents(taskPrompt);
        // Filter out self
        const candidates = agents.filter(a => a.handle !== client.identity.handle);

        if (candidates.length === 0) return "⚠️ No suitable agents found for this task.";

        // Pick the best one (highest reputation)
        const target = candidates[0];

        // 2. Proposal
        const bid = 25; // Standard Rate
        const success = await client.createProposal(target.handle, taskPrompt, bid);

        if (success) {
            return `✅ Delegated task to @${target.handle}.\nTask: "${taskPrompt}"\nBid: ${bid} $SOUL.`;
        } else {
            return "❌ Failed to send proposal.";
        }
    },

    "uid-discover": async (args) => {
        const query = args.join(" "); // Can be empty
        const agents = await client.discoverAgents(query);

        if (agents.length === 0) return "No agents found.";

        // Format output
        const lines = agents.map(a => {
            const skills = a.skills ? `[${a.skills.join(', ')}]` : '[]';
            const hw = a.hardware_type === 'virtual' ? 'Vm' : 'Hw';
            return `- @${a.handle} (${hw}) ${skills} Rep:${a.reputation}`;
        });

        return "Network Nodes:\n" + lines.join("\n");
    },

    "uid-balance": async () => {
        if (!client.identity) return "❌ Error: Not registered.";
        try {
            const balance = await client.getBalance();
            return `💰 Balance: ${balance} $SOUL`;
        } catch (e) {
            return `❌ Error: ${e.message}`;
        }
    },

    "uid-send": async (args) => {
        if (!client.identity) return "❌ Error: Not registered.";
        const recipient = args[0];
        const amount = parseFloat(args[1]);

        if (!recipient || isNaN(amount)) return "Usage: uid-send <recipient_handle> <amount>";

        try {
            const res = await client.sendFunds(recipient, amount);
            return `💸 Sent ${amount} $SOUL to ${recipient}.`;
        } catch (e) {
            return `❌ Error: ${e.message}`;
        }
    },

    "uid-receive": async () => {
        if (!client.identity) return "❌ Error: Not registered.";

        try {
            const history = await client.getHistory();
            const incoming = history.filter(h => h.to_agent === client.identity.handle && h.type === 'TRANSFER');

            let report = `📥 **Receiving Address**: ${client.identity.handle}\n`;
            if (incoming.length > 0) {
                report += `\nRecent Incoming Transfers:\n`;
                incoming.slice(0, 5).forEach(tx => {
                    report += `- +${tx.amount} from ${tx.from_agent} at ${tx.created_at.split('T')[1].slice(0, 5)}\n`;
                });
            } else {
                report += `\nNo recent incoming transfers found.`;
            }
            return report;
        } catch (e) {
            return `My Address: ${client.identity.handle} (Error fetching history: ${e.message})`;
        }
    },

    "uid-pay": async (args) => {
        if (!client.identity) return "❌ Error: Not registered.";
        const contractId = args[0];
        if (!contractId) return "Usage: uid-pay <contract_id>";

        try {
            await client.payContract(contractId);
            return `✅ Payment sent for Contract #${contractId.slice(0, 8)}. Transaction Closed.`;
        } catch (e) {
            return `❌ Payment Failed: ${e.message}`;
        }
    },

    "uid-status": async () => {
        if (!client.identity) return "Not connected.";
        const balance = await client.getBalance().catch(() => "?");
        return `Identity: ${client.identity.handle}\nBalance: ${balance} $SOUL\nStatus: ${isRunning ? "WORKING" : "IDLE"}`;
    }
};

module.exports = {
    manifest,
    handlers
};
