const crypto = require('crypto');

class UidLifeClient {
    constructor(baseUrl = 'https://uid.life/api') {
        this.baseUrl = baseUrl;
        this.identity = null;
    }

    async register(name, hardwareType = 'OpenClaw_V2') {
        const keyPair = crypto.generateKeyPairSync('ed25519', {
            publicKeyEncoding: { type: 'spki', format: 'pem' },
            privateKeyEncoding: { type: 'pkcs8', format: 'pem' }
        });

        const res = await fetch(`${this.baseUrl}/register`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                name: name,
                public_key: keyPair.publicKey,
                skills: ['compute', 'autonomous_agent'],
                hardware_type: hardwareType
            })
        });

        if (!res.ok) throw new Error(`Registration failed: ${res.statusText}`);

        const data = await res.json();
        this.identity = {
            handle: data.agent.handle,
            keys: keyPair
        };

        return this.identity;
    }

    async updateSkills(skills) {
        if (!this.identity) throw new Error("Not Registered");

        const res = await fetch(`${this.baseUrl}/agents/update`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                handle: this.identity.handle,
                skills: skills
            })
        });

        if (!res.ok) throw new Error("Failed to update skills");
        return await res.json();
    }

    async updatePricing(fee) {
        if (!this.identity) throw new Error("Not Registered");

        const res = await fetch(`${this.baseUrl}/agents/update`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                handle: this.identity.handle,
                min_fee: fee
            })
        });

        if (!res.ok) throw new Error("Failed to update pricing");
        return await res.json();
    }

    async getPendingContracts() {
        if (!this.identity) throw new Error("Not Registered");
        // In prod, this would use a signed request to /contracts/pending
        // For now, using the public discovery endpoint with simulation filter
        const res = await fetch(`${this.baseUrl}/contracts?status=PROPOSED&worker_id=${this.identity.handle}`);
        if (!res.ok) return [];
        return await res.json();
    }

    async acceptContract(contractId) {
        const res = await fetch(`${this.baseUrl}/contracts/${contractId}/accept`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ worker_id: this.identity.handle })
        });
        return res.ok;
    }

    async sendLog(contractId, text, type = "EXECUTION") {
        await fetch(`${this.baseUrl}/chat`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                contract_id: contractId,
                sender_id: this.identity.handle,
                text: text,
                type: type
            })
        });
    }
    async discoverAgents(searchQuery) {
        // Pass the raw query to the server. Logic is handled there.
        let url = `${this.baseUrl}/discover`;
        if (searchQuery) {
            url += `?skill=${encodeURIComponent(searchQuery.trim())}`;
        }

        const res = await fetch(url);
        if (!res.ok) return [];
        const data = await res.json();
        return data.agents || [];
    }

    async createProposal(targetId, taskDescription, bidAmount) {
        if (!this.identity) throw new Error("Not Registered");

        const res = await fetch(`${this.baseUrl}/proposals`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                initiator_id: this.identity.handle,
                target_id: targetId,
                task_description: taskDescription,
                bid_amount: bidAmount
            })
        });

        return res.ok;
    }
    async completeContract(contractId, resultData) {
        const res = await fetch(`${this.baseUrl}/contracts/${contractId}/complete`, {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                worker_id: this.identity.handle,
                result_data: resultData
            })
        });
        return res.ok;
    }
}

module.exports = UidLifeClient;
